## Template component

In this tutorial, we will learn how to create a template component. A template component is a component that can be rendered multiple times with different props from a data list. It is a way to create component in large list or data table.

Template component is a standard component with prop overrides, the overrides come from data list.

### Template Component Container

Template components must be placed in a template component container.

Currently only ```mui.DataFlexBox``` and ```mui.DataGrid``` support template component childs.

This tutorial only covers ```mui.DataFlexBox```, ```mui.DataGrid``` will be covered in later tutorials.

### Basic Usage

The data used by template container must be list of dict, with a unique id for each item.

**Note**: the virtualization (only render visible components in container) of ```DataFlexBox``` is enabled by default if number of data list is larger than 100. you can set ```virtualized``` to override this behavior.

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout
import random
def generate_data():
    dataList = []
    for i in range(1000):
        dataList.append({
            "id": f"{i}",
            "name": f"name{i}",
            "progress": random.random() * 100,
        })
    return dataList

class App:
    @mark_create_layout
    def my_layout(self):
        # you may need to specify the id key, default is id.
        data_box = mui.DataFlexBox(mui.HBox([
            # use set_override_props to set prop from data list
            # to override
            mui.Typography().set_override_props(value="name"),
            mui.CircularProgress().prop(variant="determinate")
                .set_override_props(value="progress"),
        ]).prop(alignItems="center")).prop(idKey="id", flexFlow="column") 
        data_box.props.dataList = generate_data()
        return mui.VBox([
            data_box.prop(flex=1, width="100%", height="100%"),
        ]).prop(flex=1)

```

### Change Data Item

When controlled input components are used in template component, we can't use controlled state anymore, the state (value for most input components) isn't valid. so we must handle ```Change``` event by ourselves.

The ```DataFlexBox``` and ```DataGrid``` provide a convenient way to handle ```Change``` event. we can use ```bind_prop``` to register a pre-defined handler for ```Change``` event. the handler will update the data item in data list.

**WARNING**: for debounced components / events such as ```TextField```/```Input``` and ```Slider```, they must be uncontrolled component (```value``` must be ```undefined```, use ```defaultValue``` instead)

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout
import random
def generate_data():
    dataList = []
    for i in range(1000):
        dataList.append({
            "id": f"{i}",
            "name": f"name{i}",
            "progress": random.random() * 100,
        })
    return dataList

class App:
    @mark_create_layout
    def my_layout(self):
        input_template = mui.Input("Name")
        # you may need to specify the id key, default is id.
        data_box = mui.DataFlexBox(mui.HBox([
            # use set_override_props to set prop from data list
            # to override
            input_template,
            mui.CircularProgress().prop(variant="determinate")
                .set_override_props(value="progress"),
        ]).prop(alignItems="center")).prop(idKey="id", flexFlow="column") 
        data_box.bind_prop(input_template, "name")
        self.datas = generate_data()
        data_box.props.dataList = self.datas
        data_box.event_item_changed.on_standard(self._on_data_change)
        return mui.VBox([
            data_box.prop(flex=1, width="100%", height="100%"),
        ]).prop(flex=1)

    async def _on_data_change(self, event: mui.Event):
        print(event)
        indexes = event.indexes
        assert not isinstance(indexes, mui.Undefined)
        print(self.datas[indexes[0]])

```

We can also update data in backend by using ```update_datas_in_index```.

```Python

from tensorpc.flow import mui, three, plus, appctx, mark_create_layout
import random
def generate_data():
    dataList = []
    for i in range(10):
        dataList.append({
            "id": f"{i}",
            "name": f"name{i}",
            "progress": random.random() * 100,
        })
    return dataList

class App:
    @mark_create_layout
    def my_layout(self):
        input_template = mui.Input("Name")
        # you may need to specify the id key, default is id.
        data_box = mui.DataFlexBox(mui.HBox([
            # use set_override_props to set prop from data list
            # to override
            input_template,
            mui.CircularProgress().prop(variant="determinate")
                .set_override_props(value="progress"),
        ]).prop(alignItems="center")).prop(idKey="id", flexFlow="column") 
        data_box.bind_prop(input_template, "name")
        self.datas = generate_data()
        data_box.props.dataList = self.datas
        data_box.event_item_changed.on_standard(self._on_data_change)
        self.data_box = data_box
        return mui.VBox([
            mui.Button("Update Random", self._on_update_random),
            data_box.prop(flex=1, width="100%", height="100%"),
        ]).prop(flex=1)

    async def _on_data_change(self, event: mui.Event):
        print(event)
        indexes = event.indexes
        assert not isinstance(indexes, mui.Undefined)
        print(self.datas[indexes[0]])

    async def _on_update_random(self):
        random_idx = random.randint(0, len(self.datas) - 1)
        await self.data_box.update_data_in_index(random_idx, {
            "progress": random.random() * 100,
        })
```

### Nested Template Component Container

You can also use template component container in template component container. the data must be nested too.

it's usually used for list of attributes such as tags. We also need nested container when we use master detail in data grid.

**WARNING**: nested template component container don't support ```bind_prop``` and ```update_datas_in_index```, you need to handle nested data update by yourself.

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout
import random
def generate_data():
    dataList = []
    for i in range(10000):
        dataList.append({
            "id": f"{i}",
            "name": f"name{i}",
            "progress": random.random() * 100,
            "tags": [
                {
                    "id": f"{j}",
                    "name": f"tag-{j}",
                } for j in range(random.randint(1, 3))
            ]
        })
    return dataList

class App:
    @mark_create_layout
    def my_layout(self):
        # you may need to specify the id key, default is id.
        data_box = mui.DataFlexBox(mui.HBox([
            # use set_override_props to set prop from data list
            # to override
            mui.Typography().set_override_props(value="name"),
            mui.CircularProgress().prop(variant="determinate")
                .set_override_props(value="progress"),
            mui.DataFlexBox(mui.Chip().prop(size="small").set_override_props(label="name")).set_override_props(dataList="tags")
            .prop(flexFlow="row")
        ]).prop(alignItems="center")).prop(idKey="id", flexFlow="column", virtualized=True) 
        data_box.props.dataList = generate_data()
        return mui.VBox([
            data_box.prop(flex=1, width="100%", height="100%"),
        ]).prop(flex=1)

```

